/* SCE CONFIDENTIAL
* PlayStation(R)3 Programmer Tool Runtime Library 430.001
* Copyright (C) 2012 Sony Computer Entertainment Inc.
* All Rights Reserved.
*/

#ifndef _SCE_IDS_H
#define _SCE_IDS_H
#include <Np.h>
#include <iostream>
#include <string>

#include <list>
#include <map>
#include "event.h"

namespace sce {
	namespace Toolkit{
		namespace NP{

	/// @brief
	/// A simple class for an %NP Communication ID.
	///
	/// This class simply wraps up an %NP Communication ID in case the structure for
	/// these IDs varies on different platforms. Currently, only the PlayStation3 is supported
	/// but further platforms will be added for later releases.
	class CommunicationId {
	public:
		/// @brief
		/// A constructor that takes the components of the %NP Communication ID as <c>std::string</c> objects.
		///
		/// A constructor that takes the components of the %NP Communication ID as <c>std::string</c> objects.
		/// It copies the first 9 characters of the ID string into the data string. If the
		/// string isn't 9 characters long, it will create an empty data structure.
		///
		/// @param id The string representing the %NP Communication ID.
		/// @param pass The string representing the passphrase for the %NP Communication ID.
		/// @param sig The signature for the %NP Communication ID.
		CommunicationId(const std::string& id, const std::string& pass, const std::string& sig) :
			m_id(),
			m_phrase(),
			m_signature()
		{
			memset(&m_id, 0, sizeof(SceNpCommunicationId));
			memset(&m_phrase, 0, sizeof(SceNpCommunicationPassphrase));
			memset(&m_signature, 0, sizeof(SceNpCommunicationSignature));
			// copies the first 9 characters into the SceNpCommunicationsId array of data.
			if(id.length() >= 9) {
				id.copy(m_id.data, 9, 0);
				m_id.term = '\0';
			}

			// should put in more validation of the id here.
			if(pass.length() >= SCE_NP_COMMUNICATION_PASSPHRASE_SIZE)
				pass.copy((char*)m_phrase.data, SCE_NP_COMMUNICATION_PASSPHRASE_SIZE, 0);

			if(sig.length() >= SCE_NP_COMMUNICATION_SIGNATURE_SIZE)
				sig.copy((char*)m_signature.data, SCE_NP_COMMUNICATION_SIGNATURE_SIZE, 0);
		}

		/// @brief
		/// A constructor that takes the components of the %NP Communication ID as their underlying objects.
		///
		/// A constructor that takes the components of the %NP Communication ID as their underlying objects.
		/// It copies contents directly from the parameters passed in.
		///
		/// @param _id		The %NP Communication ID.
		/// @param _pp		The passphrase for the %NP Communication ID.
		/// @param _sig		The signature for the %NP Communication ID.
		CommunicationId(const SceNpCommunicationId& _id,const SceNpCommunicationPassphrase& _pp ,const SceNpCommunicationSignature& _sig) :
			m_id(_id), 
			m_phrase(_pp),
			m_signature(_sig) {}


		/// @brief 
		/// The default constructor. 
		///
		/// The default constructor. 
		/// This just initializes the components of the %NP Communication ID to 0 using <c>memset()</c>.
		CommunicationId() :
			m_id(),
			m_phrase(),
			m_signature()
		{
			memset(&m_id, 0, sizeof(SceNpCommunicationId));
			memset(&m_phrase, 0, sizeof(SceNpCommunicationPassphrase));
			memset(&m_signature, 0, sizeof(SceNpCommunicationSignature));
		}

		/// @brief
		/// Gets the %NP Communication ID as a pointer to the wrapped data.
		///
		/// Gets the %NP Communication ID as a pointer to the wrapped data.
		///
		/// @return	A pointer to the %NP Communication ID.
		const SceNpCommunicationId *getId() const {return &m_id;}

		/// @brief
		/// Gets the passphrase for the %NP Communication ID.
		///
		/// Gets the passphrase for the %NP Communication ID.
		///
		/// @return		The passphrase for the %NP Communication ID.
		const SceNpCommunicationPassphrase *getPass() const {return &m_phrase;}

		/// @brief
		/// Gets the signature for the %NP Communication ID.
		///
		/// Gets the signature for the %NP Communication ID.
		///
		/// @return		The signature for the %NP Communication ID.
		const SceNpCommunicationSignature *getSig() const {return &m_signature;}

		/// @brief
		/// Less than comparison operator needed for some storage types.
		///
		/// Returns the result of a less than of comparison of the std::string representations of the %NP Communication ID.
		/// @param rhs		The %NP Communication ID to compare against.
		/// @return			If the supplied %NP Communication ID is greater than the stored ID a value of true is returned. A value of false is returned if this is not the case.
		bool operator< (const CommunicationId& rhs) const {	return std::string(this->m_id.data) < std::string(rhs.m_id.data);}
	private:
		SceNpCommunicationId			m_id;
		SceNpCommunicationPassphrase	m_phrase;
		SceNpCommunicationSignature		m_signature;
	};


	/// @brief
	/// Wraps the string being used as a %NP Service ID for commerce, ticketing, etc.
	///
	/// Wraps the string being used as a %NP Service ID for commerce, ticketing, etc.
	class ServiceId {
	public:
		/// @brief
		/// The default constructor.
		///
		/// The default constructor.
		ServiceId() {}

		/// @brief
		/// The constructor of ServiceId.
		///
		/// The constructor of ServiceId. Sets the %NP Service ID.
		/// @param _id A string representing the %NP Service ID.
 		ServiceId(const std::string& _id) : m_id(_id) {}
	
		/// @brief
		/// Sets the %NP Service ID of this object.
		///
		/// Sets the %NP Service ID of this object.
		/// @param _id A string representing the new %NP Service ID.
		void setId(const std::string& _id) {m_id = _id;}

		/// @brief
		/// Returns the %NP Service ID as a string.
		///
		/// Returns the %NP Service ID as a string.
		/// @return The %NP Service ID as a string reference.
		const std::string& getId() const {return m_id;}
	private:
		std::string m_id;
	};//end class


	/// @brief
	/// A class for managing the different IDs used inside the %NP %Toolkit library.
	///
	/// This class will store multiple %NP Communication IDs and multiple %NP Service IDs
	/// to help represent being able to use multiple services within a single application.
	/// For most cases, using one %NP Communication ID and/or one %NP Service ID is usual.
	/// The use of multiple %NP Communication IDs
	/// tends to be for a single specific use such as regional matching.
	class IdDatabase {
	public:
		/// @brief
		/// The IdDatabase Constructor.
		///
		/// This constructor takes a CommunicationId object as it's parameter as any application using
		/// the PSN will be using an %NP Communication ID. In the future, it may be necessary to provide a Service ID
		/// equivalent constructor for games only using ticketing or commerce.
		///
		/// @param initialId				A CommunicationId object.
		/// @param commerceServiceId		A commerceServiceId object.
		/// @param ticketingServiceId		A ticketingServiceId object.
		IdDatabase(const CommunicationId& initialId, const ServiceId& commerceServiceId, const ServiceId& ticketingServiceId);
	
		/// @brief
		/// The IdDatabase destructor.
		///
		/// The IdDatabase constructor.
		/// Uses the default destructors of its members.
		~IdDatabase();

		/// @brief
		/// Gets the main %NP Communication ID assumed to be used by all services.
		/// 
		/// Gets the main %NP Communication ID assumed to be used by all services.
		/// @return The %NP Communication ID.
		const CommunicationId					getPrimaryCommsId();
		/// @brief
		/// Gets a list of all %NP Communication IDs used for a particular service.
		/// 
		/// Gets a list of all %NP Communication IDs used for a particular service.
		/// @param service The service to request the IDs for.
		/// @return A list of all the %NP Communication IDs used for a particular service.
		const std::list<CommunicationId>		getCommsIdsForService(ServiceType service);
		/// @brief
		/// Adds an %NP Communication ID for a particular service type.
		///
		/// Adds an %NP Communication ID for a particular service type.
		/// @param commsId The CommunicationId object to add.
		/// @param service The service to set it against.
		void	addCommsId(const CommunicationId& commsId, ServiceType service);

		/// @brief
		/// Gets the commerce Service ID used by the application.
		///
		/// Gets the commerce Service ID used by the application.
		/// @return The commerce Service ID.
		const ServiceId&	getCommerceServiceId();

		/// @brief
		/// Gets the ticketing Service ID used by the application.
		///
		/// Gets the ticketing Service ID used by the application.
		/// @return The ticketing Service ID.
		const ServiceId&	getTicketingServiceId();

		/// @brief
		/// Sets the commerce Service ID used by the application.
		///
		/// Sets the commerce Service ID used by the application.
		/// @param id The Service ID to be set as the commerce Service ID for the title.
		void setCommerceServiceId(const ServiceId& id);

		/// @brief
		/// Sets the ticketing Service ID used by the application.
		///
		/// Sets the ticketing Service ID used by the application.
		/// @param id The Service ID to be set as the ticketing Service ID for the title.
		void setTicketingServiceId(const ServiceId& id);
	private:
		std::multimap<CommunicationId, ServiceType>		m_commsIds;
		ServiceId										m_serviceId_commerce;
		ServiceId										m_serviceId_ticketing;
	};

		}//end NP
	}//end Toolkit
}//end sce
#endif
